/**
 * \file           DnsAutoDiscovery.cpp
 * \brief          Source for CDnsAutoDiscovery Class
 * \author         RJen
 * \date           12/02/2011
 **/

#include <string.h>
#include <ctype.h>
#include "nutypedefs.h"
#include "DnsAutoDiscovery.h"
#include "ethernet.h"
#include "autodiscovery.h"
#include "os.h"
#include "memorymanager.h"
#include "hardware.h"
#include "field_debug.h"
#include "errors.h"
#include "console.h"

// Hostnames to resolve to determine whether the DNS server gives back valid IPs for invalid hostnames
// Names are provided in /Docs/Standards/Ethernet and IP Protocols/CDS Name Resolution - DNS vs Crestron Auto-Discovery
const char FirstNameToResolve [] = "invalid.crestron.com";
const char SecondNameToResolve [] = "C5B825D8-3CF4-4D25";

UINT32 IpV4DhcpDnsPriServer = 0;
UINT32 IpV4DhcpDnsSecServer = 0;

// Global pointer to the default interface
CDnsAutoDiscovery* g_pDnsAutoDiscovery = NULL;

// Used to make sure IP address is valid
extern "C" BOOL IsIpAddressValid(void);

/**
  *
  * \author      RJen
  * \brief       Constructor for the CDnsAutoDiscovery Class
  * \detail      Sets the member values to default
  * \date        12/02/2011
  * \param       none
  * \return      none
  * \retval      none
  *
  */
CDnsAutoDiscovery::CDnsAutoDiscovery(void)
{
  // Default values
  m_CurrentResolutionMode = HOST_NAME_RESOLUTION_MODE_UNKNOWN;
  m_hInitTaskHandle = NULL;
  m_hAccessLock = NULL;
  m_pDomainName = NULL;
  m_DomainNameLen = 0;
  m_bInvalidEthParams = false;
}

/**
  *
  * \author      RJen
  * \brief       Destructor for the CDnsAutoDiscovery Class
  * \detail      Destroys all member objects
  * \date        12/02/2011
  * \param       none
  * \return      none
  * \retval      none
  *
  */
CDnsAutoDiscovery::~CDnsAutoDiscovery(void)
{
  // Destroy any member objects that are allocated
  if (m_hInitTaskHandle != NULL)
  {
    OsDeleteAdvTask(m_hInitTaskHandle);
    m_hInitTaskHandle = NULL;
  }

  if (m_pDomainName != NULL)
  {
    MemMgr->FreeBlock(m_pDomainName);
    m_pDomainName = NULL;
  }

  if (m_hAccessLock != NULL)
  {
    OsDeleteLock(m_hAccessLock);
    m_hAccessLock = NULL;
  }
}

/**
  *
  * \author      RJen
  * \brief       Initializes CDnsAutoDiscovery Class
  * \detail      Sets up all resources used by the class, should never fail!
  * \date        12/02/2011
  * \param       none
  * \return      bool
  * \retval      true - success, false - failure
  *
  */
bool CDnsAutoDiscovery::Initialize(void)
{
  // Make sure extended ethernet parameters and memory manager exist
  if ( (MemMgr == NULL) || (pExtndEthParam == NULL) )
    return false;

  OsCreateLock(&m_hAccessLock); // Create the access lock

  // Start ethernet init task
  m_hInitTaskHandle = OsCreateNamedAdvTask(InitTaskWrapper, OsGetDefaultStackVarCnt(), (UINT32*)this, 0, "DnsInit");

  return true;
}

/**
 * \author    RJen
 * \brief     SendToHostQueueTaskWrapper - Wrapper for calling queue task function
 * \detail    Wrapper used inorder to create task that runs class methods. Will never
              return untill class is destroyed.
 * \date      12/02/2011
 * \param     Param: pointer to instance of the class that created the task
 * \return    none
 * \retval    none
**/
void CDnsAutoDiscovery::InitTaskWrapper(UINT32 Param)
{
  // Call function task for handling queued data
  ((CDnsAutoDiscovery*)Param)->InitTask();
}

/**
 * \author    RJen
 * \brief     Task for initializing the class object
 * \detail    Sets the domain name and hostname resolution order
 * \date      12/02/2011
 * \param     none
 * \return    none
 * \retval    none
 * \note      AutoDiscovery should be initialized before calling this function
**/
void CDnsAutoDiscovery::InitTask(void)
{
  // Wait for valid IP address (DHCP and static), prevents setting domain name before DHCP is done
  while (!IsIpAddressValid())
    HwDelayMsec(100);

  // If DHCP is disabled, set the DNS server IPs manually
  if (!IsDHCPEnabled())
  {
    NetSetDnsIp((UINT8*)&pEtherParam->pri_DNS, PRIMARY_DNS);
    NetSetDnsIp((UINT8*)&pExtndEthParam->SecondDns, SECONDARY_DNS);
  }

  // Static DNS servers could be used while in DHCP mode
  else
  {
    // If either primary or secondary DNS server is set, then use static servers
    if ( (pEtherParam->pri_DNS != 0) || (pExtndEthParam->SecondDns != 0) )
    {
      NetSetDnsIp((UINT8*)&pEtherParam->pri_DNS, PRIMARY_DNS);
      NetSetDnsIp((UINT8*)&pExtndEthParam->SecondDns, SECONDARY_DNS);
    }
  }

  // Domain name is not set, read it from NVRAM copy
  if (m_pDomainName == NULL)
    SetDomainName(pExtndEthParam->DomainName, sizeof(pExtndEthParam->DomainName));

  // Deteremine whether to use DNS or AutoDiscovery first for resolving host names
  if (m_pDomainName)
    DetermineNameResolutionMode();

  OsDeleteAdvTask(m_hInitTaskHandle); // Terminate the task
  m_hInitTaskHandle = NULL;
}

/**
  *
  * \author      RJen
  * \brief       Determines which host name resolution method to use first
  * \detail      Either use DNS or AutoDiscovery for hostname resolution first
  * \date        12/02/2011
  * \param       none
  * \return      none
  * \retval      none
  *
  */
void CDnsAutoDiscovery::DetermineNameResolutionMode(void)
{
  UINT8 TempIpAddress[IPV4_ADDR_LEN] = {0}; // Temp address for storing IP address

  // If the user specifies AutoDiscovery mode, then just set the current mode
  if (pExtndEthParam->NameResolutionMode != HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY)
  {
    // Resolve the two invalid host names; if they resolve, DNS server cannot be trusted thus use AutoDiscovery
    // Make sure to resolve the second host name without adding on the domain name
    if ( (GetHostByNameViaDns((char*)FirstNameToResolve, sizeof(FirstNameToResolve)-1, TempIpAddress)) ||
         (NetGetHostByName((char*)SecondNameToResolve, (UINT32*)TempIpAddress) == 0) )
      m_CurrentResolutionMode = HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY;
    else
      m_CurrentResolutionMode = HOST_NAME_RESOLUTION_MODE_DNS;
  }

  // Set host name resolution mode to AutoDiscovery
  else
    m_CurrentResolutionMode = HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY;
}

/**
  *
  * \author      RJen
  * \brief       Gets the hostname resolution mode
  * \detail      Gets the hostname resolution mode
  * \date        12/02/2011
  * \param       none
  * \return      HOST_NAME_RESOLUTION_MODE
  * \retval      Current hostname resolution mode
  *
  */
HOST_NAME_RESOLUTION_MODE CDnsAutoDiscovery::GetNameResolutionMode(void)
{
  return m_CurrentResolutionMode;
}

/**
  *
  * \author      RJen
  * \brief       Sets the hostname resolution mode
  * \detail      Sets the hostname resolution mode
  * \date        12/02/2011
  * \param       Mode - resolution mode
  * \return      none
  * \retval      none
  *
  */
void CDnsAutoDiscovery::SetNameResolutionMode(HOST_NAME_RESOLUTION_MODE Mode)
{
  m_CurrentResolutionMode = Mode;
}

/**
  *
  * \author      RJen
  * \brief       Checks to see if DnsAutoDiscovery is initialized or not
  * \detail      If the current hostname resolution mode is unknown, the object
                 is not initialized yet.
  * \date        12/02/2011
  * \param       none
  * \return      bool
  * \retval      true - initialized, false - not initialized
  *
  */
bool CDnsAutoDiscovery::Initialized(void)
{
  return !(m_CurrentResolutionMode == HOST_NAME_RESOLUTION_MODE_UNKNOWN);
}

/**
  *
  * \author      RJen
  * \brief       Get the host IP address using the host name
  * \detail      Gets host IP using either DNS or AutoDiscovery first. If using DNS
  *              first and it fails, try using AutoDiscovery next. If using AutoDiscovery first
  *              and it fails, don't bother using DNS.
  * \date        12/02/2011
  * \param       pHostName - pointer to hostname (could be fully qualified or not)
                 HostNameSize - size of the buffer holding the hostname
                 pIpAddr - pointer to buffer that will hold the IP address
                 IpAddrSize - size of the buffer holding the IP address
  * \return      bool
  * \retval      true - IP addr for host is found, false - if not
  * \note        AutoDiscovery must be initialized before calling this function
  *
  */
bool CDnsAutoDiscovery::GetHostByName(const char *pHostName, UINT16 HostNameSize,
                                      UINT8 *pIpAddr, UINT8 IpAddrSize)
{
  UINT16 HostNameLen = 0;
  bool IpFound = false;

  OsLock(m_hAccessLock); // Lock interface to prevent concurrent access

  // Make sure both the buffers passed in and the ethernet parameters are valid
  if ( (pHostName != NULL) && (pIpAddr != NULL) && (IpAddrSize >= IPV4_ADDR_LEN) &&
       ((HostNameLen = IsHostNameValid(pHostName,HostNameSize)) != 0) )
  {
    // Make sure ethernet parameters are valid, ex: DHCP reconnecting after link loss
    if (m_bInvalidEthParams != true)
    {
      switch (m_CurrentResolutionMode)
      {
      // Get host IP via auto discovery; if it fails, we do not use DNS as fallback
      case HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY:
        if (AutodiscoveryGetHostByName((char*)pHostName, (UINT32*)pIpAddr) == 0)
          IpFound = true;
        break;

      // Get host IP via DNS first
      case HOST_NAME_RESOLUTION_MODE_DNS:
        if (GetHostByNameViaDns(pHostName, HostNameLen, pIpAddr))
          IpFound = true;
        // DNS failed, try auto discovery as fallback
        else
        {
          if (AutodiscoveryGetHostByName((char*)pHostName, (UINT32*)pIpAddr) == 0)
            IpFound = true;
        }
        break;

      // If the object is not initialized yet, do nothing and return failure
      case HOST_NAME_RESOLUTION_MODE_UNKNOWN:
      default:
        break;
      }
    } // if (m_bInvalidEthParams != true)

    // Failed to get an IP, set it to 0.0.0.0
    if (IpFound != true)
      memset(pIpAddr, 0, IPV4_ADDR_LEN);

    if (IsDmFieldDebugIndexActive(DM_FIELD_ETHERNET_DEBUG_IDX))
      DmConsolePrintf("GetHostByName success: %s, hostname: %s, ip: %d.%d.%d.%d \r\n",
                      IpFound ? "true" : "false", pHostName, pIpAddr[0],pIpAddr[1],pIpAddr[2],pIpAddr[3]);

  } // Valid buffers and strings

  else
      DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_ETHERNET,ERR_NET_DNS_AUTO_DISC_NAME_RESOLUTION,0);

  OsUnlock(m_hAccessLock); // Unlock interface

  // Return IpFound
  return IpFound;
}

/**
  *
  * \author      RJen
  * \brief       Get the host IP address using hostname via DNS
  * \detail      If the hostname is not fully qualified, the domain name of the device will be attached
                 to make sure the DNS server does not reject the query
  * \date        12/02/2011
  * \param       pHostName - pointer to hostname (could be fully qualified or not)
                 HostNameSize - string length of the hostname (not buffer size)
                 pIpAddr - pointer to buffer that will hold the IP address (must be size checked before)
  * \return      bool
  * \retval      true - IP addr for host is found, false - if not
  *
  */
bool CDnsAutoDiscovery::GetHostByNameViaDns(const char *pHostName, UINT8 HostNameLen, UINT8 *pIpAddr)
{
  UINT8* pFQDN = NULL;
  UINT8* pHostNameStr = NULL;
  bool IpFound = false;

  // Check for "." to make sure the hostname passed in is a valid FQDN
  if (strchr(pHostName, '.') != NULL)
  {
    // Get host IP using FQDN
    if (NetGetHostByName((char*)pHostName, (UINT32*)pIpAddr) == 0)
      IpFound = true; // If operation is successful
  }

  // Turn the hostname into a FQDN so the DNS server can parse it
  else
  {
    // Make sure the FQDN length is less than maximum (-1 for "."),
    // then get temporary buffer for building the FQDN
    if ( ( (m_DomainNameLen + HostNameLen) < FULL_DOMAIN_NAME_MAX_CHARS ) &&
         (MemMgr != NULL) &&
         ( (pFQDN = MemMgr->GetBlock(m_DomainNameLen + HostNameLen + 2)) != NULL) ) // Include null term and "."
    {
      // Use temporary string pointer
      pHostNameStr = pFQDN;

      // Copy the hostname first
      memcpy(pHostNameStr, pHostName, HostNameLen);
      pHostNameStr += HostNameLen; // Increment pointer

      // Copy over the domain name if it's valid
      if (HostNameLen != 0)
      {
        *pHostNameStr++ = '.'; // Set dot
        memcpy(pHostNameStr, m_pDomainName, m_DomainNameLen);
        pHostNameStr += m_DomainNameLen; // Increment pointer
      }

      *pHostNameStr++ = 0x0; // NULL terminate the string

      // Get host IP using FQDN
      if (NetGetHostByName((char*)pFQDN, (UINT32*)pIpAddr) == 0)
        IpFound = true; // If operation is successful

      // Free allocated buffer
      if (pFQDN != NULL)
        MemMgr->FreeBlock(pFQDN);
    } // if valid options

    else
      DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_ETHERNET,ERR_NET_DNS_AUTO_DISC_NAME_RESOLUTION,1);
  } // if (strchr(pHostName, '.') != NULL)

  return IpFound;
}

/**
  *
  * \author      RJen
  * \brief       Checks if the hostname is valid
  * \detail      Checks to see for various parameters (ex: null ptr, wrong str length)
  * \date        12/02/2011
  * \param       pHostName - pointer to hostname
                 HostNameSize - hostname buffer size
  * \return      bool
  * \retval      0 - hostname is invalid, string length otherwise
  *
  */
UINT8 CDnsAutoDiscovery::IsHostNameValid(const char *pHostName, UINT16 HostNameSize)
{
  // Checks for null pointer
  if (pHostName == NULL)
  {
    DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_ETHERNET,ERR_NET_DNS_AUTO_DISC_HOSTNAME,BAD_PTR);
    return 0;
  }

  // Check host name string to make sure it's valid
  UINT16 HostNameLen = strnlen((char*)pHostName, HostNameSize);

  // Make sure the host name is null terminated and under the size limit
  if ( (HostNameLen > FULL_DOMAIN_NAME_MAX_CHARS) || (HostNameLen == HostNameSize) )
  {
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_NET_DNS_AUTO_DISC_HOSTNAME, INVALID_HOST_NAME);
    return 0;
  }

  // Return host name length; if the hostname string is 0 length, the calling thread will know it's invalid
  else
    return (UINT8) HostNameLen; // Max domain name len is 255, UINT8 can hold it
}

/**
  *
  * \author      RJen
  * \brief       Set the domain name for the class
  * \detail      Checks to see for various parameters (ex: null ptr, wrong str length)
  * \date        12/02/2011
  * \param       pDomainName - pointer to domain name
                 HostNameSize - size of the buffer holding the domain name
  * \return      none
  * \retval      none
  *
  */
void CDnsAutoDiscovery::SetDomainName(const char *pDomainName, UINT16 DomainNameSize)
{
  // Check the domain name string passed in
  UINT8 DomainNameLen = CDnsAutoDiscovery::IsHostNameValid(pDomainName, DomainNameSize);

  // Must be a valid host name
  if (DomainNameLen != 0)
  {
    // Make sure memory manager is up
    if (MemMgr != NULL)
    {
      // If domain name is already set, reset it
      if (m_pDomainName != NULL)
      {
        MemMgr->FreeBlock(m_pDomainName);
        m_DomainNameLen = 0;
      }

      // Get buffer to hold domain name
      if ( (m_pDomainName = MemMgr->GetBlock(DomainNameLen+1)) != NULL)
      {
        // Set member domain name
        memcpy(m_pDomainName, pDomainName, DomainNameLen);
        m_pDomainName[DomainNameLen] = 0x00; // Null terminate the string

        // Set member domain name len
        m_DomainNameLen = DomainNameLen;
      } // if ( (m_pDomainName = MemMgr->GetBlock(DomainNameLen+1)) != NULL)

      // Print error if a buffer cannot not be allocated
      else
        DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_NET_DNS_AUTO_DISC_HOSTNAME, UNABLE_TO_ALLOCATE_MEMORY);

    } // if (MemMgr != NULL)

    // Print error if memory manager is not up
    else
      DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_NET_DNS_AUTO_DISC_HOSTNAME, MEM_MANAGER_DOWN);

  } // if (DomainNameLen != 0)

  // Print error if an invalid domain name is passed in
  else
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_NET_DNS_AUTO_DISC_HOSTNAME, INVALID_DOMAIN_NAME);

  // Ethernet parameters are valid again since this function is called
  if (IsDHCPEnabled())
    m_bInvalidEthParams = false;
}

/**
  *
  * \author      RJen
  * \brief       Called when the ethernet link is lost
  * \detail      Sets the invalid ethernet parameter flag since the device might be moved
  * \            to another subnet; only done in DHCP mode.
  * \date        12/02/2011
  * \param       none
  * \return      none
  * \retval      none
  *
  */
void CDnsAutoDiscovery::SignalLinkLoss(void)
{
  // If in DHCP mode, ethernet parameters are no longer valid. Must wait untill DHCP reconnects
  if (IsDHCPEnabled())
    m_bInvalidEthParams = true;
}

/**
  *
  * \author      RJen
  * \brief       Flush DNS cache
  * \detail      Locks the interface and then flush
  * \date        12/02/2011
  * \param       none
  * \return      none
  * \retval      none
  *
  */
void CDnsAutoDiscovery::FlushDnsCache(void)
{
  OsLock(m_hAccessLock); // Lock interface
  NetFlushDnsCache(); // Flush cache
  OsUnlock(m_hAccessLock); // Unlock interface
}

/**
  *
  * \author      RJen
  * \brief       Called by DHCP thread in LWIP stack to set the domain name
  * \detail      Sets the domain name recieved via DHCP offer, must be called only by DHCP
  * \date        12/02/2011
  * \param       pDomainName - pointer to domain name
  * \            DomainNameSize - size of the buffer holding the domain name
  * \return      none
  * \retval      none
  *
  */
void NetDhcpSetDomainName(const char *pDomainName, UINT16 DomainNameSize)
{
  // Make sure domain name string is not zero length
  int StrLen = strnlen(pDomainName, DomainNameSize);

  // If the DHCP receives a domain name that is larger than what ethernet parameters can handle or is invalid,
  // just print an error and leave the domain name parameter alone
  if ( (StrLen >= sizeof(EXTND_ETH_PARAM::DomainName)) || (StrLen == 0) )
  {
    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_NET_DNS_AUTO_DISC_HOSTNAME, INVALID_DOMAIN_NAME);
    return;
  }

  // Set the domain name for pExtndEthParam if it exists
  if (pExtndEthParam != NULL)
  {
    // Copy the string
    memcpy(pExtndEthParam->DomainName, pDomainName, StrLen);

    // String passed in is not null terminated, terminate it to prevent errors
    pExtndEthParam->DomainName[StrLen] = 0x00; // Null terminate the string

    // Force convert hostname to all upper case due to autodsicovery
    LocalConvertEntireStringToUpper(pExtndEthParam->DomainName);

    // Set the domain name for g_pDnsAutoDiscovery if it exists
    if (g_pDnsAutoDiscovery != NULL)
      g_pDnsAutoDiscovery->SetDomainName(pExtndEthParam->DomainName, sizeof(pExtndEthParam->DomainName));
  }
}

/**
  *
  * \author      RJen
  * \brief       Command for showing DNS server addresses
  * \detail      Displays DNS server addresses
  * \date        12/20/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 ListDnsCmd(UINT32 argc, char* cmd)
{
  // If there are no arguments, then just print out the dns servers
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    // Make sure the ethernet parameters are initialized (should never be uninitialized)
    if ( (pEtherParam != NULL) && (pExtndEthParam != NULL) )
    {
      // Print DNS servers
      DmConsolePrintf("  DNS Server(s):\r\n");
      DmConsolePrintf("\t");
      DisplayDnsServer(PRIMARY_DNS);
      DmConsolePrintf("\t");
      DisplayDnsServer(SECONDARY_DNS);
    }
    else
      DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("LISTDNS\r\n");
    DmConsolePrintf("\tshows current DNS servers - no parameters needed\r\n");
  }

  return 0;
}

/**
  *
  * \author      RJen
  * \brief       Command for adding static DNS server addresses
  * \detail      Adds static DNS server to the DNS list and saves it in NVL
  * \date        12/20/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 AddDnsCmd(UINT32 argc, char* cmd)
{
  // No arguments is the same as command help
  if ( (cmd == NULL) || (*cmd == NULL) ||  (*cmd == '?'))
  {
    DmConsolePrintf("ADDDNS ip_address\r\n");
    DmConsolePrintf("\tip_address - IP address in dot decimal notation\r\n");
    return 0;
  }

  // Parse arguments
  else
  {
    // Make sure the ethernet parameters are initialized (should never be uninitialized)
    if ( (pEtherParam == NULL) || (pExtndEthParam == NULL) )
    {
      DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
      return 0;
    }

    UINT32 DnsServerIp = GetIpV4AddrFromString(cmd);

    // Check for valid IP and conver it to useable value
    if (DnsServerIp != 0)
    {
      // Check if the address is already on the list
      if ( (DnsServerIp == pEtherParam->pri_DNS) || (DnsServerIp == pExtndEthParam->SecondDns) )
        DmConsolePrintf("Address specified is already entered.\r\n");

      // Add DNS server only if there were no servers saved already
      else if (pEtherParam->pri_DNS == 0)
      {
        pEtherParam->pri_DNS = DnsServerIp;
        TriggerSystemSave(); // Save the setting in NVL
      }

      else if (pExtndEthParam->SecondDns == 0)
      {
        pExtndEthParam->SecondDns = DnsServerIp;
        TriggerSystemSave(); // Save the setting in NVL
      }

      // If both IPs are set already
      else
        DmConsolePrintf("Max number of static DNS servers (%d) already defined.\r\n", MAX_DNS_SERVERS);

      return 0;
    } // if (DnsServerIp != 0)

    DmConsolePrintf("invalid address\r\n"); // Display error if it's an invalid address
  } // // Parse arguments
  return 0;
}

/**
  *
  * \author      RJen
  * \brief       Command for removing static DNS server addresses
  * \detail      Removes static DNS server from the DNS list and saves it in NVL
  * \date        12/20/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 RemDnsCmd(UINT32 argc, char* cmd)
{
  // No arguments is the same as command help
  if ( (cmd == NULL) || (*cmd == NULL) ||  (*cmd == '?') )
  {
    DmConsolePrintf("ADDDNS ip_address\r\n");
    DmConsolePrintf("\tip_address - IP address in dot decimal notation\r\n");
    return 0;
  }

  // Parse IP addr
  else
  {
    // Make sure the ethernet parameters are initialized (should never be uninitialized)
    if ( (pEtherParam == NULL) || (pExtndEthParam == NULL) )
    {
      DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
      return 0;
    }

    UINT32 DnsServerIp = GetIpV4AddrFromString(cmd);

    // Check for valid IP and conver it to useable value
    if (DnsServerIp != 0)
    {
      // Matches the IP and then removes it
      if (pEtherParam->pri_DNS == DnsServerIp)
      {
        // If the secondary DNS server is set, just move it to primary
        if (pExtndEthParam->SecondDns != 0)
        {
          pEtherParam->pri_DNS = pExtndEthParam->SecondDns;
          pExtndEthParam->SecondDns = 0;
        }

        // Just reset the primary DNS if the scondary is not set
        else
          pEtherParam->pri_DNS = 0;

        TriggerSystemSave(); // Save the setting in NVL
      } // if (pEtherParam->pri_DNS == DnsServerIp)

      // IP addr matched, just reset
      else if (pExtndEthParam->SecondDns == DnsServerIp)
      {
        pExtndEthParam->SecondDns = 0;
        TriggerSystemSave(); // Save the setting in NVL
      }

      else
      {
        if ( (pEtherParam->pri_DNS == 0) && (pExtndEthParam->SecondDns == 0) )
          DmConsolePrintf("No DNS Servers to delete");
        else
          DmConsolePrintf("Address specified is not in the static DNS server list");
      }

      return 0; // Valid address
    } // if (DnsServerIp != 0)

    DmConsolePrintf("invalid address\r\n"); // Display error if it's an invalid address
  } // // Parse arguments

  return 0;
}

/**
  *
  * \author      RJen
  * \brief       Command for testing DNS server
  * \detail      Depending on the parameters passed in, it will either print or set domain name
  * \date        11/23/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 TestDnsCmd(UINT32 argc, char* cmd)
{
  // Must have arguments
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    DmConsolePrintf("Invalid domain name entered, please enter valid host name.\r\n");
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("TESTDNS string\r\n");
    DmConsolePrintf("\tstring - ASCII string containing host name\r\n");
  }

  // Parse string argument
  else
  {
    UINT16 bLen = 0;

    // Ensure valid host name
    for ( char* cp = cmd; (*cp != NULL) && (bLen < (FULL_DOMAIN_NAME_MAX_CHARS+1)); cp++)
    {
      if ( !isalnum(*cp) && *cp != '-' &&  *cp != '.')
      {
        DmConsolePrintf("ERROR: Illegal character in host name.(A-Z,a-z,0-9,-,.)");
        return 0;
      }
      bLen++;
    }

    if (bLen >= (FULL_DOMAIN_NAME_MAX_CHARS+1))
    {
      DmConsolePrintf("ERROR: Host name cannot be > %d characters", FULL_DOMAIN_NAME_MAX_CHARS);
      return 0;
    }

    // Make sure name resolution interface is initialized before resolving the host name
    if ( (g_pDnsAutoDiscovery != NULL) && (g_pDnsAutoDiscovery->Initialized()) )
    {
      UINT8 IpAddress[IPV4_ADDR_LEN] = {0,0,0,0};

      // Get host IP address
      if (g_pDnsAutoDiscovery->GetHostByNameViaDns(cmd, bLen-1, IpAddress) != false)
        DmConsolePrintf("IPAddress = %d.%d.%d.%d\r\n", IpAddress[0],IpAddress[1],IpAddress[2],IpAddress[3]);
      else
        DmConsolePrintf("\tStatus: Failure\r\n");
    }
    else
      DmConsolePrintf("DNS is not initialized yet, please retry later.\r\n");
  }

  return 0;
}


/**
  *
  * \author      RJen
  * \brief       Command for changing domain name
  * \detail      Depending on the parameters passed in, it will either print or set domain name
  * \date        11/23/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 DomainNameCmd(UINT32 argc, char* cmd)
{
  // Make sure the ethernet parameters are initialized (should never be uninitialized)
  if (pExtndEthParam == NULL)
  {
    DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
    return 0;
  }

  // If there are no arguments, then just print out the domain name
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    // Make sure domain name string is not zero length and is null temrinated
    int StrLen = strnlen(pExtndEthParam->DomainName, sizeof(pExtndEthParam->DomainName));

    if ( (StrLen != 0) || (StrLen != sizeof(pExtndEthParam->DomainName)) )
      DmConsolePrintf("Domain Name: %s\r\n",pExtndEthParam->DomainName);
    else
      DmConsolePrintf("Domain Name: \r\n");
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("DOMAINNAME [string | /CLEAR ]:\r\n");
    DmConsolePrintf("\tstring - ASCII string containing domain name\r\n");
    DmConsolePrintf("\t/CLEAR - clears the value\r\n");
    DmConsolePrintf("\tNo parameter - current value\r\n");
  }

  // Command with clear argument
  else if ( !strncmp(cmd,"/CLEAR",6) )
  {
    memset(pExtndEthParam->DomainName, 0, sizeof(pExtndEthParam->DomainName));
    DmConsolePrintf("Domain name cleared. Reboot to take effect.\r\n");
    TriggerSystemSave();
  }

  // Parse string argument
  else
  {
    UINT16 bLen = 0;
    for ( char* cp = cmd; (*cp != NULL) && (bLen < sizeof(pExtndEthParam->DomainName)); cp++)
    {
      if ( !isalnum(*cp) && *cp != '-' &&  *cp != '.')
      {
        DmConsolePrintf("ERROR: Illegal character in domain name.(A-Z,a-z,0-9,-,.)");
        return 0;
      }
      bLen++;
    }
    if ( bLen >= sizeof(pExtndEthParam->DomainName) )
    {
      DmConsolePrintf("ERROR: Domain name cannot be > %d characters",sizeof(pExtndEthParam->DomainName)-1);
      return 0;
    }

    // Force convert hostname to all upper case due to autodsicovery
    LocalConvertEntireStringToUpper(cmd);

    // Save domain name in NVL
    memcpy(pExtndEthParam->DomainName, cmd, bLen);
    pExtndEthParam->DomainName[bLen] = 0x00; // Null terminate the string
    TriggerSystemSave();
    DmConsolePrintf("New domain name set. Reboot to take effect.\r\n");
  }

  return 0;
}

/**
  *
  * \author      RJen
  * \brief       Command for changing domain name
  * \detail      Depending on the parameters passed in, it will either print or set domain name
  * \date        11/23/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 DhcpOptionsCmd(UINT32 argc, char* cmd)
{
  // Make sure the ethernet parameters are initialized (should never be uninitialized)
  if (pExtndEthParam == NULL)
  {
    DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
    return 0;
  }

  // If there are no arguments, then just print out the domain name
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    if (pExtndEthParam->DhcpOptions == DHCP_OPTIONS_HOSTNAME)
      DmConsolePrintf("Current DHCP Discover Request Option: HOSTNAME (Type 12)\r\n");
    else
      DmConsolePrintf("Current DHCP Discover Request Option: FQDN (Type 81)\r\n");
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("DHCPOPT [HOSTNAME | FQDN]\r\n");
    DmConsolePrintf("\tHOSTNAME - Send Local HostName in DHCP Discover Request - Default Option\r\n");
    DmConsolePrintf("\tFQDN - Send the Fully Qualified Domain Name in DHCP Discover Request\r\n");
    DmConsolePrintf("\tNo parameter - displays current setting\r\n");
  }

  // Check command arguments
  else if ( !stricmp(cmd,"HOSTNAME") )
  {
    pExtndEthParam->DhcpOptions = DHCP_OPTIONS_HOSTNAME;
    DmConsolePrintf("New DHCP Discovery Request Option set.  Reboot to take effect.\r\n");
    TriggerSystemSave();
  }

    // Check command arguments
  else if ( !stricmp(cmd,"FQDN") )
  {
    pExtndEthParam->DhcpOptions = DHCP_OPTIONS_FQDN;
    DmConsolePrintf("New DHCP Discovery Request Option set.  Reboot to take effect.\r\n");
    TriggerSystemSave();
  }

  // Parse string argument
  else
  {
    DmConsolePrintf("ERROR: Invalid parameter");
  }

  return 0;
}

/**
  *
  * \author      RJen
  * \brief       Command for changing host name resolution mode
  * \detail      Changes or prints the name resolution mode
  * \date        12/02/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 ResolutionModeCmd(UINT32 argc, char* cmd)
{
  // Make sure name resolution interface is initialized
  if ( (g_pDnsAutoDiscovery == NULL) || (!g_pDnsAutoDiscovery->Initialized()) )
  {
    DmConsolePrintf("Hostname resolution interface is not initialized yet, please retry later.\r\n");
    return 0;
  }

  // Make sure the ethernet parameters are initialized (should never be uninitialized)
  if ( (pEtherParam == NULL) || (pExtndEthParam == NULL) )
  {
    DmConsolePrintf("Ethernet parameters are not initialized yet\r\n");
    return 0;
  }

  // If there are no arguments, then just print out the name resolution mode
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    switch (g_pDnsAutoDiscovery->GetNameResolutionMode())
    {

      case HOST_NAME_RESOLUTION_MODE_DNS:
        DmConsolePrintf("Using DNS Server followed by Autodiscovery for the hostname resolution.\r\n");
        break;
      case HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY:
        DmConsolePrintf("Using Autodiscovery for hostname resolution.\r\n");
        break;
      case HOST_NAME_RESOLUTION_MODE_UNKNOWN:
      default:
        DmConsolePrintf("Hostname Resolution Method not defined yet, please retry later\r\n");
        break;
    }
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("HostNameResOrder [USE_DNS | USE_AUTO]\r\n");
    DmConsolePrintf("\tUSE_DNS - Use DNS first for hostname resolution (Default Option)\r\n");
    DmConsolePrintf("\tUSE_AUTO - Use AutoDiscovery for hostname resolution\r\n");
    DmConsolePrintf("\tNo parameter - displays current setting\r\n");
  }

  // Parse string argument
  else if ( !stricmp(cmd,"USE_DNS") )
  {
    // Update the current mode and then save it to NVL
    g_pDnsAutoDiscovery->SetNameResolutionMode(HOST_NAME_RESOLUTION_MODE_DNS);
    pExtndEthParam->NameResolutionMode = HOST_NAME_RESOLUTION_MODE_DNS;
    TriggerSystemSave();
    DmConsolePrintf("New hostname resolution mode set. Reboot to take effect.\r\n");
  }

  else if ( !stricmp(cmd,"USE_AUTO") )
  {
    g_pDnsAutoDiscovery->SetNameResolutionMode(HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY);
    pExtndEthParam->NameResolutionMode = HOST_NAME_RESOLUTION_MODE_AUTODISCOVERY;
    TriggerSystemSave();
    DmConsolePrintf("New hostname resolution mode set. Reboot to take effect.\r\n");
  }

  else
  {
    DmConsolePrintf("ERROR: Invalid parameter");
  }

  return 0;
}


/**
  *
  * \author      RJen
  * \brief       Command for resolving a inputted hostname
  * \detail      Resolves hostname using g_pDnsAutoDiscovery
  * \date        12/02/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 ResolveHostnameCmd(UINT32 argc, char* cmd)
{
  // Must have arguments
  if ( (cmd == NULL) || (*cmd == NULL) )
  {
    DmConsolePrintf("Invalid hostname entered, please enter valid hostname.\r\n");
    return 0;
  }

  // Command help
  if (*cmd == '?')
  {
    DmConsolePrintf("RSLVHOSTNAME [HOSTNAME]:\r\n");
    DmConsolePrintf("\tResolves the specified hostname using the DNS Server and then Autodiscovery \r\n");
    DmConsolePrintf("\tHOSTNAME - Host name to resolve\r\n");
  }

  // Parse string argument
  else
  {
    UINT16 bLen = 0;

    // Ensure valid host name
    for ( char* cp = cmd; (*cp != NULL) && (bLen < sizeof(EXTND_ETH_PARAM::DomainName)); cp++)
    {
      if ( !isalnum(*cp) && *cp != '-' &&  *cp != '.')
      {
        DmConsolePrintf("ERROR: Illegal character in hostname.(A-Z,a-z,0-9,-,.)");
        return 0;
      }
      bLen++;
    }

    if (bLen >= sizeof(EXTND_ETH_PARAM::DomainName))
    {
      DmConsolePrintf("ERROR: Hostname name cannot be > %d characters", sizeof(EXTND_ETH_PARAM::DomainName)-1);
      return 0;
    }

    // Make sure name resolution interface is initialized
    if ( (g_pDnsAutoDiscovery != NULL) && (g_pDnsAutoDiscovery->Initialized()) )
    {
      UINT8 IpAddress[IPV4_ADDR_LEN] = {0,0,0,0};

      // Get host IP address
      if (g_pDnsAutoDiscovery->GetHostByName(cmd, bLen+1, IpAddress, sizeof(IpAddress)) != false)
        DmConsolePrintf("IPAddress = %d.%d.%d.%d\r\n", IpAddress[0],IpAddress[1],IpAddress[2],IpAddress[3]);
      else
        DmConsolePrintf("Failed to get IPAddress\r\n");
    }
    else
      DmConsolePrintf("Hostname resolution interface is not initialized yet, please retry later.\r\n");
  }

  return 0;
}